/*
 * File Name: nfc.c
 *
 * Description: Contains functions to handle processing the NFC/RFID
 * stack.
 *
 * Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
 *
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

//===============================================================

#include "nfc.h"
#include "MFDESFireAESAuth.h"

//===============================================================
//		Global Variables
//===============================================================

static uint8_t pui8SessionKey[16] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

// This is the Random Number A (RndA) that will be used during the AES authentication sequence.
// It is hard coded in this example, however a Random Number Generator (RNG) could be used to generate the number instead.
static uint8_t pui8RndA[16] = {0x79, 0xd4, 0x66, 0x29,0xf7,0xe1,0x12,0xc3,0x79, 0xd4, 0x66, 0x29,0xf7,0xe1,0x12,0xc3};

// This is the AES key that will be used for the AES authentication process. Modify this key with the Nfc_setAESKey() function.
static uint8_t pui8AESkey[16] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

//===============================================================

//===============================================================
// NAME: void Nfc_runAesAuth(void)
//
// BRIEF: Runs through the NFC stack for DESFire EV1
//
// INPUTS: None
//
// OUTPUTS: None
//
// PROCESS:	[1] Turn RF field on
//			[2] Set up TRF for initiator
//			[3] Call Anticollision function for tag detection
//			[4] Call Layer 4 commands
//			[5] If a tag is activated, go through AES Auth process
//
//===============================================================

void
Nfc_runAesAuth(void)
{
	uint8_t 	ui8AuthSuccess = STATUS_FAIL;

	Trf797xTurnRfOn();				// Enable RF Field for TRF7970

	Trf797xWriteInitiatorSetup();	// Send initial command to configure Type 4A Reader/Writer

	IRQ_ON;							// Enable interrupts on I/O pin for the IRQ

	// When a PICC is exposed to an unmodulated operating field
	// it shall be able to accept a quest within 5 ms.
	// PCDs should periodically present an unmodulated field of at least
	// 5.1 ms duration. (ISO14443-3)
	McuDelayMillisecond(6);

	Iso14443aAnticollision(REQA);	// Do a complete anticollision sequence as described
									// in ISO14443-3 standard for type A

	if(Iso14443aLayer4() == STATUS_SUCCESS)
	{
		LED_14443A_ON;		//Layer 4 successful.

		ui8AuthSuccess = Aes_authenticate(&pui8SessionKey[0], &pui8RndA[0], &pui8AESkey[0]);
		if (ui8AuthSuccess == STATUS_SUCCESS)
		{
#ifdef ENABLE_UART
			UartSendCString("MIFARE DESFire EV1 AES Authentication Complete");
			UartPutNewLine();
			UartPutNewLine();
#endif
//===================================================================================
// This section is an example of a user specific application that will allow for the
// contents of an DFEV1 Tag to be read out after Authentication
// The functions used below must be uncommented in the iso14443a.c file as well.
// Additional memory usage is ~1400 bytes Flash (depending on optimization settings)
//===================================================================================

//			uint8_t	pui8OffsetBuf[3] = {0x00,0x00,0x00};
//
//			Iso14443aDFEV1GetAppId(0);
//			Iso14443aDFEV1SelectApp(1);
//			if (Iso14443aDFEV1GetFileId(0) == STATUS_SUCCESS)
//			{
//				Iso14443aDFEV1ReadData(0x01, &pui8OffsetBuf[0], 0x1F, 1);
//				Iso14443aDFEV1ReadData(0x02, &pui8OffsetBuf[0], 0x6C, 0);
//			}

			McuDelayMillisecond(200);	// Arbitrary delay between each authentication to simulate the run time of a user defined application
		}
		LED_14443A_OFF;
		LED_15693_OFF;
	}
	else
	{
		// Layer 4 fail.
		LED_14443A_OFF;
		LED_15693_OFF;
	}

	IRQ_OFF;

	Trf797xTurnRfOff();			// Turns off the RF Field

	Trf797xResetIrqStatus();	// Reads IRQ to ensure it has been cleared

	// Poll every 50 milliseconds
	McuDelayMillisecond(50);
}

//===============================================================
// NAME: void Nfc_setAesKey(const uint8_t * pui8AESKey, uint8_t ui8AESKeyLength)
//
// BRIEF: Used to change the AES key used for Authentication
//
// INPUTS:
//	Parameters:
//		uint8_t		ui8AESKeyLength		Length of the AES key
//		uint8_t		*pui8AESKey			Pointer to array containing AES key
//
// OUTPUTS: None
//
// PROCESS:	[1] Check if AES key length is equal to 16
//			[2] Copy Key array
//
//===============================================================


void Nfc_setAesKey(const uint8_t * pui8AESKey, uint8_t ui8AESKeyLength)
{
	uint8_t i = 0;

	if (ui8AESKeyLength == 16)
	{
		for (i = 0; i < ui8AESKeyLength; i++)
		{
			pui8AESkey[i] = pui8AESKey[i];
		}
	}
	else
	{
		// Wrong Length, Do Nothing
	}
}

